/***************************************************************************
 *
 * Copyright 2010,2011 BMW Car IT GmbH
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include "Layermanager.h"
#include "Configuration.h"
#include "SignalHandler.h"
#include "Log.h"
#include <unistd.h>
#include <sys/time.h>
#include <fcntl.h>
#include <semaphore.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

/*defined here because should never be visible throw the header file of the LayerManager
 * all of the common headers between the client and LayerManager are available in the sysroot*/
#define LM_INIT_SEM "/lmanager_init"


int main(int argc, char **argv)
{
    // monitor startup performance
    unsigned int startupTimeInMs = 0;
    timeval start;
    timeval end;
    mode_t old_umask;
    sem_t* lm_init_sem = NULL;
    int numRetries = 400;
    int waiting_time = 5*1000;/* 5 ms*/

    gettimeofday(&start, 0);

    /* change the mask, safe to do it here , no thread
     * no other files are affected by this change*/
    old_umask = umask(0);
    /* create a semaphore with value 0
     * every client will block till LayerManager post the semaphore*/
    lm_init_sem = sem_open(LM_INIT_SEM, O_CREAT, 0666, 0);
    while (lm_init_sem == SEM_FAILED || lm_init_sem == NULL)
    {
        if ((errno == EACCES) && (numRetries > 0))
        {
            LOG_WARNING("Gal2dGraphicSystem",
                       "could not OPEN a lmanager sem, retry cnt: "<< numRetries);
            usleep(waiting_time);
            numRetries--;
            lm_init_sem = sem_open(LM_INIT_SEM, O_CREAT, 0666, 0);
        }
        else
        {
            log_to_errmem("LAYERMANAGER: FAILED TO CREATE lmanager SEMAPHORE,"
                    "errno: %d ,rety_cnt: %d\n",errno,numRetries);
            LOG_ERROR("main","FAILED TO CREATE lmanager SEMAPHORE, "
                      "GIVING UP... SORRY,errno: "<<errno<<" rety cnt: "<< numRetries);
            return EXIT_FAILURE;
        }
    }
    umask(old_umask);

    if (lm_init_sem == NULL)
    {
        return EXIT_FAILURE;
    }
    // collect all configuration settings
    Configuration configuration(argc, argv);

    // setup logging
    Log::getInstance()->setConsoleLogLevel((LOG_MODES)configuration.getLogLevelConsole());
    Log::getInstance()->setFileLogLevel((LOG_MODES)configuration.getLogLevelFile());
    Log::getInstance()->setDLTLogLevel((LOG_MODES)configuration.getLogLevelTrace());

    LOG_WARNING("main",
             "Starting Layermanager (version: " << ILM_VERSION << ")");

    // log configuration options
    configuration.logAllSettings();

    {
        SignalHandler signalHandler;
        Layermanager layermanager(configuration);

        if (layermanager.startManagement())
        {
            gettimeofday(&end, 0);
            startupTimeInMs = ((end.tv_sec - start.tv_sec) * 1000) + ((end.tv_usec - start.tv_usec) / 1000);
            LOG_WARNING("main",
                     "Startup complete. "
                     "EnterMainloop, time=" << startupTimeInMs << "ms"
                     <<" plugin cnt: ren: "<<layermanager.getRendererCnt()
                     <<", com: "<<layermanager.getCommunicatorCnt()
                     <<", scn: "<<layermanager.getSceneproviderCnt()
                     <<", config: "<<layermanager.getConfiguratorCnt());

            /*LayerManager is ready to use, post the semaphore*/
            sem_post(lm_init_sem);

            signalHandler.waitForShutdownSignal();

            LOG_WARNING("main", "Stopping service.");
            layermanager.stopManagement();
        }
        else
        {
            /*check the at least detected plugins*/
            log_to_errmem("LAYERMANAGER: service fail to start,"
                    "count plugins:ren:%d,com%d,scn:%d\n",
                    layermanager.getRendererCnt(),
                    layermanager.getCommunicatorCnt(),
                    layermanager.getSceneproviderCnt());

            LOG_ERROR("main", "LayerManager service fail to start.");
        }
        sem_close(lm_init_sem);
        sem_unlink(LM_INIT_SEM);
    }

    LOG_WARNING("main", "Shutdown complete.");
    Log::closeInstance();
    return EXIT_SUCCESS;
}

